const canvas = document.querySelector('canvas');
const context = canvas.getContext('2d');

canvas.width = 1024;
canvas.height = 576;

context.fillRect(0, 0, canvas.width, canvas.height);

const gravity = 0.7;

// Check if we have dynamic character/mob data
const useDynamicData = typeof window.ARENA3_DATA !== 'undefined' && window.ARENA3_DATA;
const basePath = useDynamicData ? window.ARENA3_DATA.basePath : '';

// Helper function to build sprite paths
function buildSpritePaths(spriteData, state) {
    if (!useDynamicData || !spriteData || !spriteData.sprites || !spriteData.sprites[state]) {
        return null;
    }
    const spriteState = spriteData.sprites[state];
    if (spriteState && spriteState.images) {
        // img paths from getPngSequence start with /, basePath ends with /
        // So we need to remove trailing slash from basePath or handle it
        const cleanBasePath = basePath.replace(/\/$/, '');
        return spriteState.images.map(img => cleanBasePath + img);
    }
    return null;
}

// Helper function to get framesMax
function getFramesMax(spriteData, state) {
    if (!useDynamicData || !spriteData || !spriteData.sprites || !spriteData.sprites[state]) {
        return 1;
    }
    return spriteData.sprites[state].framesMax || 1;
}

// Use dynamic base path if available, otherwise use relative path
const backgroundPath = useDynamicData ? window.ARENA3_DATA.backgroundPath : './backgrounds/Background1.png';
const shopPath = useDynamicData ? window.ARENA3_DATA.shopPath : './backgrounds/shop_anim.png';

const background = new Sprite({
    position: {
        x: 0,
        y: 0
    },
    imageSrc: backgroundPath,
    scale: 1,
    isBackground: true // Flag to identify background
})

const shop = new Sprite({
    position: {
        x: 700,
        y: 170
    },
    imageSrc: shopPath,
    scale: 2.5,
    framesMax: 6
})

// Build player sprites dynamically or use defaults
const playerIdleImages = buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'idle') || [
    './farmer-peasant/idle/01.png',
    './farmer-peasant/idle/02.png',
    './farmer-peasant/idle/03.png',
    './farmer-peasant/idle/04.png',
    './farmer-peasant/idle/05.png'
];

const playerSprites = {
    idle: {
        images: playerIdleImages,
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'idle') || 5
    },
    run: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'run') || playerIdleImages,
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'run') || 5
    },
    jump: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'jump') || [playerIdleImages[0]],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'jump') || 1
    },
    fall: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'fall') || [playerIdleImages[0]],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'fall') || 1
    },
    attack1: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'attack1') || [
            './farmer-peasant/fight/01.png',
            './farmer-peasant/fight/02.png',
            './farmer-peasant/fight/03.png',
            './farmer-peasant/fight/04.png',
            './farmer-peasant/fight/05.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'attack1') || 5
    },
    takeHit: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'takeHit') || [
            './farmer-peasant/hurt/01.png',
            './farmer-peasant/hurt/02.png',
            './farmer-peasant/hurt/03.png',
            './farmer-peasant/hurt/04.png',
            './farmer-peasant/hurt/05.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'takeHit') || 5
    },
    death: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.character : null, 'death') || [
            './farmer-peasant/lose/01.png',
            './farmer-peasant/lose/02.png',
            './farmer-peasant/lose/03.png',
            './farmer-peasant/lose/04.png',
            './farmer-peasant/lose/05.png',
            './farmer-peasant/lose/06.png',
            './farmer-peasant/lose/07.png',
            './farmer-peasant/lose/08.png',
            './farmer-peasant/lose/09.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.character : null, 'death') || 9
    }
};

// Get character stats from data or use defaults
const characterStats = useDynamicData && window.ARENA3_DATA.character && window.ARENA3_DATA.character.stats 
    ? window.ARENA3_DATA.character.stats 
    : { hp: 100, attack: 50, defense: 20 };

const player = new Fighter({
    position: {
        x: 100,
        y: 451  // Positioned on the floor (494 - 200 + 157 offset)
    },
    velocity: {
        x: 0,
        y: 0
    },
    offset: {
        x: 0,
        y: 0
    },
    images: playerIdleImages,
    framesMax: playerSprites.idle.framesMax,
    scale: 2.5,
    offset: {
        x: 215,
        y: 157
    },
    sprites: playerSprites,
    health: characterStats.hp,
    attack: characterStats.attack,
    defense: characterStats.defense,
    attackBox: {
        offset: {
            x: 100,
            y: 50
        },
        width: 160,
        height: 50
    }
});

// Ensure player starts with idle sprite
player.switchSprite('idle');

// Build enemy sprites dynamically or use defaults
const enemyIdleImages = buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'idle') || [
    './desert-rat/idle/left/01.png',
    './desert-rat/idle/left/02.png',
    './desert-rat/idle/left/03.png',
    './desert-rat/idle/left/04.png',
    './desert-rat/idle/left/05.png',
    './desert-rat/idle/left/06.png',
    './desert-rat/idle/left/07.png',
    './desert-rat/idle/left/08.png',
    './desert-rat/idle/left/09.png'
];

const enemySprites = {
    idle: {
        images: enemyIdleImages,
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'idle') || 9
    },
    run: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'run') || enemyIdleImages,
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'run') || 9
    },
    jump: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'jump') || [enemyIdleImages[0]],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'jump') || 1
    },
    fall: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'fall') || [enemyIdleImages[0]],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'fall') || 1
    },
    attack1: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'attack1') || [
            './desert-rat/attack/left/01.png',
            './desert-rat/attack/left/02.png',
            './desert-rat/attack/left/03.png',
            './desert-rat/attack/left/04.png',
            './desert-rat/attack/left/05.png',
            './desert-rat/attack/left/06.png',
            './desert-rat/attack/left/07.png',
            './desert-rat/attack/left/08.png',
            './desert-rat/attack/left/09.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'attack1') || 9
    },
    takeHit: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'takeHit') || [
            './desert-rat/idle/left/01.png',
            './desert-rat/idle/left/02.png',
            './desert-rat/idle/left/03.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'takeHit') || 3
    },
    death: {
        images: buildSpritePaths(useDynamicData ? window.ARENA3_DATA.mob : null, 'death') || [
            './desert-rat/idle/left/01.png',
            './desert-rat/idle/left/02.png',
            './desert-rat/idle/left/03.png',
            './desert-rat/idle/left/04.png',
            './desert-rat/idle/left/05.png',
            './desert-rat/idle/left/06.png',
            './desert-rat/idle/left/07.png'
        ],
        framesMax: getFramesMax(useDynamicData ? window.ARENA3_DATA.mob : null, 'death') || 7
    }
};

// Get mob stats from data or use defaults
const mobStats = useDynamicData && window.ARENA3_DATA.mob && window.ARENA3_DATA.mob.stats 
    ? window.ARENA3_DATA.mob.stats 
    : { hp: 100, attack: 40, defense: 20 };

const enemy = new Fighter({
    position: {
        x: 800,
        y: 464  // Positioned on the floor (494 - 200 + 170 offset)
    },
    velocity: {
        x: 0,
        y: 0
    },
    color: 'blue',
    offset: {
        x: 0,
        y: 0
    },
    images: enemyIdleImages,
    framesMax: enemySprites.idle.framesMax,
    scale: 2.5, // Same scale as player (will be 200px)
    offset: {
        x: 215,
        y: 170
    },
    sprites: enemySprites,
    health: mobStats.hp,
    attack: mobStats.attack,
    defense: mobStats.defense,
    attackBox: {
        offset: {
            x: -170,
            y: 50
        },
        width: 170,
        height: 50
    }
})

// Ensure enemy starts with idle sprite and is visible immediately
enemy.switchSprite('idle');

// Force enemy to be visible by ensuring images are set
if (enemy.images && enemy.images.length > 0) {
    // Images are already set - check if they're loaded
    const firstImage = enemy.images[0];
    console.log('✅ Enemy images loaded:', enemy.images.length, '| First image complete:', firstImage ? firstImage.complete : false);
    
    // If images aren't loaded yet, wait for them
    if (firstImage && !firstImage.complete) {
        firstImage.onload = function() {
            console.log('✅ Enemy first image now loaded!');
            // Force redraw
            enemy.switchSprite('idle');
        };
    }
} else if (enemy.image) {
    // Single image sprite sheet
    console.log('✅ Enemy image loaded (sprite sheet) | Complete:', enemy.image.complete);
    if (!enemy.image.complete) {
        enemy.image.onload = function() {
            console.log('✅ Enemy sprite sheet now loaded!');
        };
    }
} else {
    console.warn('⚠️ Enemy has no images! Attempting to load...');
    // Force load idle images
    if (enemy.sprites && enemy.sprites.idle) {
        if (enemy.sprites.idle.images && enemy.sprites.idle.images.length > 0) {
            enemy.images = enemy.sprites.idle.images;
            enemy.image = enemy.sprites.idle.images[0];
            enemy.framesMax = enemy.sprites.idle.framesMax;
            console.log('✅ Enemy images forced loaded from sprites');
        }
    }
}

// Log initialization
console.log('🎮 Arena 3 Game Initialized');
console.log('👤 Player:', {
    name: useDynamicData ? window.ARENA3_DATA.character.name : 'Default',
    health: player.health,
    attack: player.attackPower,
    defense: player.defense,
    position: player.position
});
console.log('👹 Enemy:', {
    name: useDynamicData ? window.ARENA3_DATA.mob.name : 'Default',
    health: enemy.health,
    attack: enemy.attackPower,
    defense: enemy.defense,
    position: enemy.position,
    sprite: enemy.image ? 'loaded' : 'not loaded',
    images: enemy.images ? enemy.images.length + ' images' : 'no images'
});

// Automatic fighting system - no keyboard controls needed
let currentRound = 1;
let isPlayerTurn = true;
let roundInProgress = false;
let roundCooldown = 0;
let attackCooldown = 0;
let attackExecuted = false;
let returningToPosition = false;
const ROUND_DELAY = 90; // frames between rounds (~1.5 seconds at 60fps) - faster
const ATTACK_DELAY = 15; // frames before attack executes - faster
const ATTACK_DISTANCE = 300; // Distance to maintain for attacks (increased to get closer)
const ATTACK_LUNGE_DISTANCE = 80; // How far forward to move when attacking

// Store starting positions
const playerStartX = 100;
const enemyStartX = 800;
// Return positions (closer to center, not all the way back)
const playerReturnX = 250; // Player returns to this position (not all the way left)
const enemyReturnX = 750; // Enemy returns to this position (not all the way right)

decreaseTimer();

// Track current state for header display (declared outside animate function)
let playerCurrentState = 'idle';
let enemyCurrentState = 'idle';

// Initialize health bars
setTimeout(() => {
    if (player && player.maxHealth) {
        const playerHealthPercent = (player.health / player.maxHealth) * 100;
        gsap.set('#playerHealth', { width: playerHealthPercent + '%' });
    }
    if (enemy && enemy.maxHealth) {
        const enemyHealthPercent = (enemy.health / enemy.maxHealth) * 100;
        gsap.set('#enemyHealth', { width: enemyHealthPercent + '%' });
    }
}, 100);

function handleAutomaticFighting() {
    // Don't fight if someone is dead
    if (player.dead || enemy.dead) {
        return;
    }

    // Handle round cooldown
    if (roundCooldown > 0) {
        roundCooldown--;
        return;
    }

    const attacker = isPlayerTurn ? player : enemy;
    const defender = isPlayerTurn ? enemy : player;
    const attackerStartX = isPlayerTurn ? playerStartX : enemyStartX;
    const attackerReturnX = isPlayerTurn ? playerReturnX : enemyReturnX;
    const defenderStartX = isPlayerTurn ? enemyStartX : playerStartX;
    
    // Calculate distance between fighters
    const distance = Math.abs(attacker.position.x - defender.position.x);
    
    // Log round start
    if (!roundInProgress && distance <= ATTACK_DISTANCE) {
        console.log(`⚔️ Round ${currentRound} - ${isPlayerTurn ? 'Player' : 'Enemy'} turn`);
    }
    
    // If returning to position after attack
    if (returningToPosition) {
        const distanceToReturn = Math.abs(attacker.position.x - attackerReturnX);
        if (distanceToReturn > 5) {
            // Move back to return position (not all the way to starting position)
            if (attacker.position.x < attackerReturnX) {
                attacker.velocity.x = 8; // Faster return
            } else if (attacker.position.x > attackerReturnX) {
                attacker.velocity.x = -8; // Faster return
            }
        } else {
            // Reached return position
            attacker.position.x = attackerReturnX;
            attacker.velocity.x = 0;
            returningToPosition = false;
            isPlayerTurn = !isPlayerTurn;
            roundInProgress = false;
            roundCooldown = ROUND_DELAY;
            currentRound++;
            attackExecuted = false;
            console.log(`🔄 Turn switched - Now ${isPlayerTurn ? 'Player' : 'Enemy'} turn | Round ${currentRound}`);
        }
        defender.velocity.x = 0;
        return;
    }
    
    // If too far, move closer to center
    if (distance > ATTACK_DISTANCE) {
        roundInProgress = false;
        attackExecuted = false;
        attackCooldown = 0;
        
        if (isPlayerTurn) {
            // Player moves right toward center
            if (player.position.x < enemy.position.x - ATTACK_DISTANCE) {
                player.velocity.x = 8; // Faster movement
            } else {
                player.velocity.x = 0;
            }
            enemy.velocity.x = 0;
        } else {
            // Enemy moves left toward center
            if (enemy.position.x > player.position.x + ATTACK_DISTANCE) {
                enemy.velocity.x = -8; // Faster movement
            } else {
                enemy.velocity.x = 0;
            }
            player.velocity.x = 0;
        }
    } else {
        // Close enough, perform attack
        if (!roundInProgress) {
            roundInProgress = true;
            attackCooldown = ATTACK_DELAY;
            attackExecuted = false;
        }
        
        // Calculate minimum distance to maintain (don't pass through opponent)
        // Player gets very close when attacking, enemy stays a bit further
        const minDistance = isPlayerTurn ? 40 : 80; // Player gets very close (40px) vs enemy (80px)
        const currentDistance = Math.abs(attacker.position.x - defender.position.x);
        
        // Move forward during attack
        if (attackCooldown > 0) {
            attackCooldown--;
            // Start moving forward, but stop if too close
            if (currentDistance > minDistance) {
                if (isPlayerTurn) {
                    attacker.velocity.x = 15; // Faster attack movement for player
                } else {
                    attacker.velocity.x = -12; // Faster attack movement
                }
            } else {
                attacker.velocity.x = 0; // Stop if too close
            }
            defender.velocity.x = 0;
        } else if (!attackExecuted) {
            // Execute attack
            if (!attacker.dead && !defender.dead && !attacker.isAttacking) {
                console.log(`🗡️ ${isPlayerTurn ? 'Player' : 'Enemy'} attacks!`);
                attacker.attack();
                attackExecuted = true;
                // Continue moving forward during attack to ensure contact, but stop if too close
                if (currentDistance > minDistance) {
                    if (isPlayerTurn) {
                        attacker.velocity.x = 14; // Faster for player to get very close
                    } else {
                        attacker.velocity.x = -10; // Faster
                    }
                } else {
                    attacker.velocity.x = 0; // Stop if too close
                }
            }
        } else {
            // Keep moving forward for a bit during attack animation to ensure hit connects
            const attackFramesMax = attacker.sprites.attack1.framesMax;
            if (attacker.frameCurrent < attackFramesMax - 1) {
                // Still in attack animation, keep moving forward but stop if too close
                // Player continues moving closer even during attack animation
                if (currentDistance > minDistance) {
                    if (isPlayerTurn) {
                        attacker.velocity.x = 12; // Keep moving closer during attack
                    } else {
                        attacker.velocity.x = -8; // Faster
                    }
                } else {
                    // If very close, slow down but don't stop completely
                    if (isPlayerTurn && currentDistance > 20) {
                        attacker.velocity.x = 6; // Slow approach when very close
                    } else {
                        attacker.velocity.x = 0; // Stop if too close
                    }
                }
            } else {
                // Attack animation done, slow down
                attacker.velocity.x *= 0.9;
            }
        }
        
        // Wait for attack animation to complete, then return to position
        const attackFramesMax = attacker.sprites.attack1.framesMax;
        const attackComplete = attackExecuted && 
            (attacker.image !== attacker.sprites.attack1.image || 
             attacker.frameCurrent >= attackFramesMax - 1);
        
        // Also wait for defender's hit animation to complete if they were hit
        const defenderHitFramesMax = defender.sprites.takeHit.framesMax;
        const hitAnimationComplete = defender.image !== defender.sprites.takeHit.image ||
            defender.frameCurrent >= defenderHitFramesMax - 1;
        
        if (attackComplete && !attacker.isAttacking && hitAnimationComplete) {
            // Start returning to starting position
            console.log(`✅ Round ${currentRound} complete - ${isPlayerTurn ? 'Player' : 'Enemy'} finished attack`);
            returningToPosition = true;
            attacker.velocity.x = 0;
        }
    }
}

let frameCount = 0;
function animate() {
    window.requestAnimationFrame(animate);
    frameCount++;
    
    context.fillStyle = 'black';
    context.fillRect(0, 0, canvas.width, canvas.height);
    background.update();
    shop.update();
    
    // Handle automatic fighting (this sets velocities)
    handleAutomaticFighting();
    
    // Log enemy visibility on first few frames
    if (frameCount <= 10) {
        const currentImg = enemy.images && enemy.images[0] ? enemy.images[0] : null;
        console.log(`Frame ${frameCount}: Enemy draw check -`, {
            hasImages: !!enemy.images,
            imagesLength: enemy.images ? enemy.images.length : 0,
            hasImage: !!enemy.image,
            currentImage: currentImg ? 'exists' : 'missing',
            imageComplete: currentImg ? currentImg.complete : false,
            imageNaturalWidth: currentImg ? currentImg.naturalWidth : 0,
            imageNaturalHeight: currentImg ? currentImg.naturalHeight : 0,
            position: enemy.position,
            offset: enemy.offset,
            drawX: enemy.position.x - enemy.offset.x,
            drawY: enemy.position.y - enemy.offset.y
        });
    }
    
    player.update();
    enemy.update();

    // Handle sprite animations based on velocity and state
    if (player.isAttacking) {
        // Don't change sprite during attack
        if (playerCurrentState !== 'attack1') {
            playerCurrentState = 'attack1';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('attack1');
            }
        }
    } else if (player.dead) {
        if (playerCurrentState !== 'death') {
            playerCurrentState = 'death';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('death');
            }
        }
    } else if (player.velocity.x !== 0) {
        player.switchSprite('run');
        if (playerCurrentState !== 'run') {
            playerCurrentState = 'run';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('run');
            }
        }
    } else if (player.velocity.y < 0) {
        player.switchSprite('jump');
        if (playerCurrentState !== 'jump') {
            playerCurrentState = 'jump';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('jump');
            }
        }
    } else if (player.velocity.y > 0) {
        player.switchSprite('fall');
        if (playerCurrentState !== 'fall') {
            playerCurrentState = 'fall';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('fall');
            }
        }
    } else if (!player.dead) {
        player.switchSprite('idle');
        if (playerCurrentState !== 'idle') {
            playerCurrentState = 'idle';
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('idle');
            }
        }
    }

    if (enemy.isAttacking) {
        // Don't change sprite during attack
        if (enemyCurrentState !== 'attack1') {
            enemyCurrentState = 'attack1';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('attack1');
            }
        }
    } else if (enemy.dead) {
        if (enemyCurrentState !== 'death') {
            enemyCurrentState = 'death';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('death');
            }
        }
    } else if (enemy.velocity.x !== 0) {
        enemy.switchSprite('run');
        if (enemyCurrentState !== 'run') {
            enemyCurrentState = 'run';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('run');
            }
        }
    } else if (enemy.velocity.y < 0) {
        enemy.switchSprite('jump');
        if (enemyCurrentState !== 'jump') {
            enemyCurrentState = 'jump';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('jump');
            }
        }
    } else if (enemy.velocity.y > 0) {
        enemy.switchSprite('fall');
        if (enemyCurrentState !== 'fall') {
            enemyCurrentState = 'fall';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('fall');
            }
        }
    } else if (!enemy.dead) {
        enemy.switchSprite('idle');
        if (enemyCurrentState !== 'idle') {
            enemyCurrentState = 'idle';
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('idle');
            }
        }
    }

    //detect colission & enemy hit
    // Check frames 1, 2, 3, or 4 for player attack (5 frame animation)
    // Check collision more frequently during attack
    if (player.isAttacking && (player.frameCurrent >= 1 && player.frameCurrent <= 4)) {
        if (rectangularCollision({
            rectangle1: player,
            rectangle2: enemy
        })) {
            // Calculate damage using Arena 1 system
            const attackerPower = player.attackPower || 50;
            const attackerLevel = 1; // Default level
            let damage = Math.floor(attackerPower * (0.8 + Math.random() * 0.4));
            
            // Level boost
            const levelBoost = 1 + (attackerLevel * 0.05);
            damage = Math.floor(damage * levelBoost);
            
            // Critical hit chance (20%)
            let isCritical = false;
            if (Math.random() < 0.2) {
                damage = Math.floor(damage * 1.75);
                isCritical = true;
            }
            
            // Minimum damage
            damage = Math.max(damage, 5);
            
            console.log(`💥 Player hits Enemy for ${damage} damage${isCritical ? ' (CRITICAL!)' : ''} | Enemy HP: ${enemy.health} -> ${enemy.health - damage}`);
            enemy.takeHit(damage);
            player.isAttacking = false;
            // Update state to show hit
            if (typeof window.updateMobState === 'function') {
                window.updateMobState('takeHit');
                setTimeout(() => {
                    if (enemyCurrentState !== 'death') {
                        window.updateMobState(enemyCurrentState);
                    }
                }, 300);
            }
            // Update health bar based on percentage
            const healthPercent = (enemy.health / enemy.maxHealth) * 100;
            gsap.to('#enemyHealth', {
                width: healthPercent + '%'
            })
        }
    }

    // if player misses - check at end of attack animation
    if (player.isAttacking && player.frameCurrent >= 4) {
        player.isAttacking = false;
    }

    // enemy hits
    // Check frames 2, 3, 4, 5, 6, 7, or 8 for enemy attack (9 frame animation)
    // Check collision more frequently during attack
    if (enemy.isAttacking && (enemy.frameCurrent >= 2 && enemy.frameCurrent <= 8)) {
        if (rectangularCollision({
            rectangle1: enemy,
            rectangle2: player
        })) {
            // Calculate damage using Arena 1 system
            const attackerPower = enemy.attackPower || 40;
            const attackerLevel = 1; // Default level
            let damage = Math.floor(attackerPower * (0.8 + Math.random() * 0.4));
            
            // Level boost
            const levelBoost = 1 + (attackerLevel * 0.05);
            damage = Math.floor(damage * levelBoost);
            
            // Critical hit chance (20%)
            let isCritical = false;
            if (Math.random() < 0.2) {
                damage = Math.floor(damage * 1.75);
                isCritical = true;
            }
            
            // Minimum damage
            damage = Math.max(damage, 5);
            
            console.log(`💥 Enemy hits Player for ${damage} damage${isCritical ? ' (CRITICAL!)' : ''} | Player HP: ${player.health} -> ${player.health - damage}`);
            player.takeHit(damage);
            enemy.isAttacking = false;
            // Update state to show hit
            if (typeof window.updateCharacterState === 'function') {
                window.updateCharacterState('takeHit');
                setTimeout(() => {
                    if (playerCurrentState !== 'death') {
                        window.updateCharacterState(playerCurrentState);
                    }
                }, 300);
            }
            // Update health bar based on percentage
            const healthPercent = (player.health / player.maxHealth) * 100;
            gsap.to('#playerHealth', {
                width: healthPercent + '%'
            })
        }
    }

    // if enemy misses - check at end of attack animation
    if (enemy.isAttacking && enemy.frameCurrent >= 8) {
        enemy.isAttacking = false;
    }

    // end game based on health
    if (enemy.health <= 0 || player.health <= 0) {
        const winner = player.health > enemy.health ? 'Player' : 'Enemy';
        console.log(`🏆 Game Over! ${winner} wins! | Player HP: ${player.health} | Enemy HP: ${enemy.health}`);
        determinewinner({ player, enemy, timerId });
    }
}

// Preload critical images (idle sprites) before starting animation
function preloadCriticalImages() {
    const criticalImages = [];
    
    // Collect player idle images (most important for initial display)
    if (player.sprites && player.sprites.idle && player.sprites.idle.images) {
        player.sprites.idle.images.forEach(img => {
            if (img) {
                criticalImages.push({img: img, type: 'player'});
            }
        });
    }
    
    // Collect enemy idle images (most important for initial display)
    if (enemy.sprites && enemy.sprites.idle && enemy.sprites.idle.images) {
        enemy.sprites.idle.images.forEach(img => {
            if (img) {
                criticalImages.push({img: img, type: 'enemy'});
            }
        });
    }
    
    console.log(`⏳ Preloading ${criticalImages.length} critical images (idle sprites)...`);
    
    if (criticalImages.length === 0) {
        console.log('✅ No critical images to preload, starting game immediately...');
        animate();
        return;
    }
    
    let loadedCount = 0;
    const totalImages = criticalImages.length;
    
    criticalImages.forEach((item, index) => {
        const img = item.img;
        if (img.complete && img.naturalWidth > 0) {
            loadedCount++;
            console.log(`✅ ${item.type} image ${index + 1}/${totalImages} already loaded`);
        } else {
            img.onload = function() {
                loadedCount++;
                console.log(`✅ ${item.type} image ${index + 1}/${totalImages} loaded (${this.naturalWidth}x${this.naturalHeight})`);
                if (loadedCount === totalImages) {
                    console.log('✅ All critical images preloaded! Starting game...');
                    // Ensure sprites are set
                    player.switchSprite('idle');
                    enemy.switchSprite('idle');
                    animate();
                }
            };
            img.onerror = function() {
                loadedCount++;
                console.warn(`⚠️ ${item.type} image ${index + 1}/${totalImages} failed to load:`, this.src);
                if (loadedCount === totalImages) {
                    console.log('✅ All critical images processed! Starting game...');
                    player.switchSprite('idle');
                    enemy.switchSprite('idle');
                    animate();
                }
            };
        }
    });
    
    // If all images are already loaded
    if (loadedCount === totalImages) {
        console.log('✅ All critical images already loaded! Starting game...');
        player.switchSprite('idle');
        enemy.switchSprite('idle');
        animate();
    }
}

// Start preloading critical images
preloadCriticalImages();

// Keyboard controls removed - game is now fully automatic